<?php

namespace App\Repositories;

use Illuminate\Support\Facades\Hash;
use Tymon\JWTAuth\Facades\JWTAuth;
use App\Repositories\Interfaces\UserRepositoryInterface;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log; // Import Log facade at the top
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use App\Models\Setting;
use Carbon\Carbon;


class UserRepository implements UserRepositoryInterface
{
    protected function logError($url, $error_message)
    {
        Log::error('Error in user repository function', [
            'url' => $url,
            'error' => $error_message
        ]);
    }

    public function encryptText($text) {
        $encrypt_key = env('ENCRYPT_KEY');
        $encrypt_code = env('ENCRYPT_CODE');
        // Encrypt the text
        $encryptedText = openssl_encrypt($text, 'AES-128-CBC', $encrypt_key, OPENSSL_RAW_DATA, $encrypt_code);
        if($encryptedText == '' || $encryptedText == null) {
            return $text;
        }
        // Return the encrypted text in Base64 encoding
        return base64_encode($encryptedText);
    }

    public function decryptText($encryptedText) {
        $encrypt_key = env('ENCRYPT_KEY');
        $encrypt_code = env('ENCRYPT_CODE');
        $text = $encryptedText;
        // Decode the text from Base64
        $encryptedText = base64_decode($encryptedText);
        // Decrypt the text
        $decryptedText = openssl_decrypt($encryptedText, 'AES-128-CBC', $encrypt_key, OPENSSL_RAW_DATA, $encrypt_code);
        if($decryptedText == '' || $decryptedText == null) {
            return $text;
        }
        return $decryptedText;
    }

    function generateRandomString($length, $type)
    {
        if (intval($type) == 1) {
            $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        } else if (intval($type) == 2) {
            $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*+-/?><';
        } else if (intval($type) == 3) {
            $characters = '0123456789';
        } else if (intval($type) == 4) {
            $characters = '0123456789abcdefg';
        } else {
            $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        }

        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    public function userRegister(array $data)
    {
        try {
            $full_name     = $data['full_name'];
            $user_name     = $data['user_name'];
            $nic_number    = $data['nic_number'];
            $contact_number = $data['contact_number'];
            $password      = $data['password'];

            $push_id       = $data['push_id'] ?? null;
            $os_type       = isset($data['os_type']) ? (int) $data['os_type'] : 0;

            // Backend rule: only allow admin type=1 else force type=2
            $role_id = isset($data['role_id']) ? (int) $data['role_id'] : 0;

            // Create user (timestamps auto, password auto-hash via cast)
            $new_user = User::create([
                'role_id'  => $role_id,
                'full_name'     => ucwords(strtolower(trim($full_name))),
                'user_name'     => trim($user_name),
                'nic_number'    => trim($nic_number),
                'contact_number' => trim($contact_number),
                'password'      => Hash::make($password),
                'profile_image' => $profile_image,
                'push_id'       => $push_id,
                'os_type'       => $os_type,
                'is_active'     => 1,
            ]);

            $token = JWTAuth::fromUser($new_user);

            return [
                'success' => true,
                'message' => 'Success',
                'data'    => [
                    'user_id' => (int) $new_user->id,
                    'token'   => $token,
                ],
                'status'  => 201,
            ];

        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());
            dd($e->getMessage());
            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
                'status'  => 500,
            ];
        }
    }

    public function userLogin(array $data)
    {
        try {
            $user_name = isset($data['user_name']) ? trim($data['user_name']) : null;
            $password  = $data['password'] ?? null;
            $push_id   = $data['push_id'] ?? null;
            $os_type   = isset($data['os_type']) ? (int) $data['os_type'] : 0;

            $user = User::where('user_name', $user_name)
                    ->where('is_active', 1)
                    ->first();

            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'Account does not exist.',
                    'data'    => null,
                    'status'  => 404,
                ];
            }

            if (!Hash::check($password, $user->password)) {
                return [
                    'success' => false,
                    'message' => 'Invalid credentials. Please check and try again.',
                    'data'    => null,
                    'status'  => 401,
                ];
            }

            $token = JWTAuth::fromUser($user);

            // Update device info (optional)
            $user->push_id = $push_id;
            $user->os_type = $os_type;
            $user->updated_at = Carbon::now();
            $user->save();

            return [
                'success' => true,
                'message' => 'User sign in success.',
                'data'    => [
                    'user_id' => (int) $user->id,
                    'token'   => $token,
                ],
                'status'  => 200,
            ];

        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
                'status'  => 500,
            ];
        }
    }

    public function userValidate(array $data)
    {
        try {
            $user_name = isset($data['user_name']) ? trim($data['user_name']) : null;

            $user = User::where('user_name', $user_name)
                    ->where('is_active', 1)->first();

            $has_user = $user ? 1 : 0;

            return [
                'success' => true,
                'status'  => 200,
                'message' => 'User verification success.',
                'data'    => [
                    'has_user' => $has_user,
                    'user_id'  => $user ? (int) $user->id : 0,
                ],
            ];

        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
                'status'  => 500,
            ];
        }
    }

    public function userData()
    {
        try {
            $user = JWTAuth::parseToken()->authenticate();

            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'User does not exist.',
                    'data'    => null,
                    'status'  => 404,
                ];
            }

            return [
                'success' => true,
                'message' => 'Success',
                'data'    => [
                    'user_id'       => (int) $user->id,
                    'user_type_id'  => (int) ($user->user_type_id ?? 0),
                    'full_name'     => $user->full_name ?? null,
                    'user_name'     => $user->user_name ?? null,
                    'nic_number'    => $user->nic_number ?? null,
                    'mobile_number' => $user->mobile_number ?? null,
                    'email_address' => $user->email_address ?? null,
                    'profile_image' => $user->profile_image ?? null,
                    'os_type'       => (int) ($user->os_type ?? 0),
                    'push_id'       => $user->push_id ?? null,
                    'is_active'     => (int) ($user->is_active ?? 0),
                ],
                'status' => 200,
            ];

        } catch (\Throwable $e) {
            $this->logError('auth/user', $e->getMessage());

            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
                'status'  => 500,
            ];
        }
    }

    public function userAll(array $data)
    {
        try {
            $user_type_id = isset($data['user_type_id']) ? (int) $data['user_type_id'] : 0;
            $search       = isset($data['search']) ? trim($data['search']) : null;
            $status       = isset($data['status']) ? (int) $data['status'] : -1;

            $page    = isset($data['page']) ? max(1, (int) $data['page']) : 1;
            $perPage = isset($data['per_page']) ? (int) $data['per_page'] : 10;
            $perPage = max(1, min($perPage, 100));

            $query = User::query();

            if ($user_type_id !== 0) {
                $query->where('user_type_id', $user_type_id);
            }

            if ($status === 0 || $status === 1) {
                $query->where('is_active', $status);
            }

            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('user_name', 'LIKE', "%{$search}%")
                    ->orWhere('full_name', 'LIKE', "%{$search}%")
                    ->orWhere('mobile_number', 'LIKE', "%{$search}%")
                    ->orWhere('nic_number', 'LIKE', "%{$search}%")
                    ->orWhere('email_address', 'LIKE', "%{$search}%");
                });
            }

            // Select only safe columns
            $users = $query->select([
                    'id',
                    'user_type_id',
                    'full_name',
                    'user_name',
                    'nic_number',
                    'mobile_number',
                    'email_address',
                    'profile_image',
                    'os_type',
                    'push_id',
                    'is_active',
                    'created_at',
                    'updated_at',
                ])
                ->orderBy('user_name', 'asc')
                ->paginate($perPage, ['*'], 'page', $page);

            return [
                'success' => true,
                'message' => 'Success',
                'status'  => 200,
                'data'    => [
                    'items' => $users->items(),
                    'meta'  => [
                        'total'        => $users->total(),
                        'per_page'     => $users->perPage(),
                        'current_page' => $users->currentPage(),
                        'last_page'    => $users->lastPage(),
                    ],
                ],
            ];

        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
                'status'  => 500,
            ];
        }
    }

    public function userManage(array $data)
    {
        try {
            $user_id = (int) $data['user_id'];

            $user = User::where('id', $user_id)->first();

            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'User does not exist.',
                    'data'    => null,
                    'status'  => 404,
                ];
            }

            $user->update([
                'full_name'     => ucwords(strtolower(trim($data['full_name']))),
                'nic_number'    => trim($data['nic_number']),
                'mobile_number' => trim($data['mobile_number']),
                'email_address' => strtolower(trim($data['email_address'])),
            ]);

            return [
                'success' => true,
                'message' => 'User details updated successfully.',
                'data'    => [
                    'user_id' => (int) $user->id,
                ],
                'status'  => 200,
            ];

        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());
            dd($e->getMessage());
            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => $e->getMessage(),
                'status'  => 500,
            ];
        }
    }

    public function userPasswordChange(array $data)
    {
        try {
            $user = JWTAuth::parseToken()->authenticate();

            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'Unauthorized.',
                    'data'    => null,
                    'status'  => 401,
                ];
            }

            $currentPassword = $data['current_password'];
            $newPassword     = $data['new_password'];

            // Check current password
            if (!Hash::check($currentPassword, $user->password)) {
                return [
                    'success' => false,
                    'message' => 'Current password is incorrect.',
                    'data'    => null,
                    'status'  => 401,
                ];
            }

            // Prevent using same password again (optional but good)
            if (Hash::check($newPassword, $user->password)) {
                return [
                    'success' => false,
                    'message' => 'New password cannot be the same as the current password.',
                    'data'    => null,
                    'status'  => 422,
                ];
            }

            // Update password (NO Hash::make because model casts to 'hashed')
            $user->password = $newPassword;
            $user->save();

            // Optional: invalidate all sessions by regenerating token (recommended)
            // JWTAuth::invalidate(JWTAuth::getToken());

            return [
                'success' => true,
                'message' => 'Password updated successfully.',
                'data'    => [
                    'user_id' => (int) $user->id,
                ],
                'status'  => 200,
            ];

        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
                'status'  => 500,
            ];
        }
    }

    public function userLogout(array $data)
    {
        try {
            $user = JWTAuth::parseToken()->authenticate();

            if (!$user) {
                return [
                    'success' => false,
                    'message' => 'Unauthorized.',
                    'data'    => null,
                    'status'  => 401,
                ];
            }

            // Invalidate current token
            JWTAuth::invalidate(JWTAuth::getToken());

            return [
                'success' => true,
                'message' => 'Logout successful.',
                'data'    => null,
                'status'  => 200,
            ];

        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
                'status'  => 500,
            ];
        }
    }

}
