<?php

namespace App\Repositories;

use App\Models\FoodCategory;
use App\Models\FoodItem;
use App\Models\FoodType;
use App\Models\MeatBreed;
use App\Models\MeatCountry;
use App\Models\MeatCutType;
use App\Repositories\Interfaces\SettingRepositoryInterface;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Setting;
use Carbon\Carbon;
use PhpParser\Node\Expr\FuncCall;

class SettingRepository implements SettingRepositoryInterface
{
    protected function logError($url, $error_message)
    {
        Log::error('Error in setting repository function', [
            'url' => $url,
            'error' => $error_message
        ]);
    }

    public function encryptText($text) {
        $encrypt_key = env('ENCRYPT_KEY');
        $encrypt_code = env('ENCRYPT_CODE');
        // Encrypt the text
        $encryptedText = openssl_encrypt($text, 'AES-128-CBC', $encrypt_key, OPENSSL_RAW_DATA, $encrypt_code);
        if($encryptedText == '' || $encryptedText == null) {
            return $text;
        }
        // Return the encrypted text in Base64 encoding
        return base64_encode($encryptedText);
    }

    public function decryptText($encryptedText) {
        $encrypt_key = env('ENCRYPT_KEY');
        $encrypt_code = env('ENCRYPT_CODE');
        $text = $encryptedText;
        // Decode the text from Base64
        $encryptedText = base64_decode($encryptedText);
        // Decrypt the text
        $decryptedText = openssl_decrypt($encryptedText, 'AES-128-CBC', $encrypt_key, OPENSSL_RAW_DATA, $encrypt_code);
        if($decryptedText == '' || $decryptedText == null) {
            return $text;
        }
        return $decryptedText;
    }

    function generateRandomString($length, $type)
    {
        if (intval($type) == 1) {
            $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        } else if (intval($type) == 2) {
            $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*+-/?><';
        } else if (intval($type) == 3) {
            $characters = '0123456789';
        } else if (intval($type) == 4) {
            $characters = '0123456789abcdefg';
        } else {
            $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        }

        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    public function foodCategoryAdd(array $data) {
        try {
            $food_category = isset($data['food_category']) ? trim($data['food_category']) : null;
            $status = isset($data['status']) ? intval($data['status']) : 0;
            $order_by = isset($data['order_by']) ? intval($data['order_by']) : 0;

            $category_exists = FoodCategory::whereRaw('LOWER(food_category) = ?', [strtolower($food_category)])->first();

            if (isset($category_exists->id)) {
                if ($category_exists->is_active == 0) {
                    $output['success'] = false;
                    $output['message'] = "Category already exists but is being deleted. food category id : " . $category_exists->id;
                    $output['data'] = null;
                    $output['status'] = 409;
                } else {
                    $output['success'] = false;
                    $output['message'] = "Category already exists.";
                    $output['data'] = null;
                    $output['status'] = 409;
                }
            } else {
                if($status != 1) {
                    $status = 0;
                }
                FoodCategory::create([
                    'food_category' => ucwords(strtolower($food_category)),
                    'order_by' => $order_by,
                    'item_count' => 0,
                    'status' => $status,
                    'is_active' => 1,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now()
                ]);

                $output['success'] = true;
                $output['message'] = "Category added successfully.";
                $output['data'] = null;
                $output['status'] = 201;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodCategoryValidate(array $data) {
        try {
            $food_category_id = isset($data['food_category_id']) ? intval($data['food_category_id']) : 0;

            $category_exists = FoodCategory::where('id', $food_category_id)->where('is_active', 1)->first();

            if (!isset($category_exists->id)) {
                $output['success'] = true;
                $output['message'] = "Category not found.";
                $output['data'] = null;
                $output['status'] = 200;
            } else {
                $output['success'] = true;
                $output['message'] = "Category found.";
                $output['data']['id'] = isset($category_exists->id) ? intval($category_exists->id) : 0;
                $output['data']['food_category'] = isset($category_exists->food_category) ? $category_exists->food_category : null;
                $output['data']['item_count'] = isset($category_exists->item_count) ? intval($category_exists->item_count) : 0;
                $output['data']['order_by'] = isset($category_exists->order_by) ? intval($category_exists->order_by) : 0;
                $output['data']['status'] = isset($category_exists->is_active) ? intval($category_exists->is_active) : 0;
                $output['status'] = 200;
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodCategoryEdit(array $data) {
        try {
            $food_category_id = isset($data['food_category_id']) ? intval($data['food_category_id']) : 0;
            $food_category = isset($data['food_category']) ? trim($data['food_category']) : null;
            $order_by = isset($data['order_by']) ? intval($data['order_by']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : -1;
            $is_active = isset($data['is_active']) ? intval($data['is_active']) : -1;
            $category_exists = FoodCategory::where('id', $food_category_id)
                                /*->where('is_active', 1 )*/
                                ->first();

            if (!isset($category_exists->id)) {
                $output['success'] = true;
                $output['message'] = "Category does not exist.";
                $output['data'] = null;
                $output['status'] = 200;
            } else {
                if ($food_category != null || $food_category != "") {
                    $category_name_exists = FoodCategory::whereRaw('LOWER(food_category) = ?', [strtolower($food_category)])
                                                        ->where('id', '!=', $food_category_id)
                                                        /*->where('is_active', 1)*/
                                                        ->exists();
                    if ($category_name_exists) {
                        $output['success'] = false;
                        $output['message'] = "Category name already exists.";
                        $output['data'] = null;
                        $output['status'] = 409;
                    } else {
                        $category_exists->food_category = ucwords(strtolower($food_category));
                        $category_exists->order_by = $order_by;
                        $category_exists->updated_at = Carbon::now();
                        if($status == 0 || $status == 1) {
                            $category_exists->status = $status;
                        }
                        if($is_active == 0 || $is_active == 1) {
                            $category_exists->is_active = $is_active;
                        }
                        $category_exists->save();
                        $output['success'] = true;
                        $output['message'] = "Category updated successfully.";
                        $output['data'] = null;
                        $output['status'] = 200;
                    }
                } else {
                    $output['success'] = false;
                    $output['message'] = "Category name is required.";
                    $output['data'] = null;
                    $output['status'] = 422;
                }

            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodCategoryDelete(array $data) {
        try {
            $food_category_id = isset($data['food_category_id']) ? intval($data['food_category_id']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : 0;

            $category_exists = FoodCategory::where('id', $food_category_id)->first();

            if (!isset($category_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Category does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $output['success'] = true;

                if ($category_exists->is_active != $status) {
                    if ($status != 1) {
                        $status = 0;
                        $output['message'] = "Category deleted successfully.";
                    } else {
                        $output['message'] = "Category activated successfully.";
                    }

                    $category_exists->is_active = $status;
                    $category_exists->save();
                } else {
                    $output['message'] = "Category is already " . ($status == 1 ? "active." : "inactive.");
                }

                $output['data'] = null;
                $output['status'] = 200;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodCategoryAll(array $data)
    {
        try {
            $search      = $data['search']      ?? null;
            $status   = isset($data['status']) ? (int)$data['status'] : -1;
            $is_active   = isset($data['is_active']) ? (int)$data['is_active'] : -1;
            $page_number = isset($data['page_number']) ? (int)$data['page_number'] : 1;
            $per_page    = isset($data['per_page']) ? (int)$data['per_page'] : 10;

            // Validate is_active (only accept 0 or 1)
            $is_active = in_array($is_active, [0, 1], true) ? $is_active : null;

            $query = FoodCategory::query()
                        ->select('id', 'food_category', 'item_count', 'order_by', 'status', 'is_active')
                        ->when(in_array($is_active, [0, 1], true), function ($q) use ($is_active) {
                            $q->where('is_active', $is_active);
                        })
                        ->when(in_array($status, [0, 1], true), function ($q) use ($status) {
                            $q->where('status', $status);
                        })
                        ->when(!empty($search), function ($q) use ($search) {
                            $q->where('food_category', 'like', "%{$search}%");
                        });

            $categories = $query
                            ->orderBy('order_by', 'asc')
                            ->orderBy('id', 'asc')
                            ->paginate($per_page, ['*'], 'page', $page_number);

            if ($categories->total() === 0) {
                $output = [
                    'success' => true,
                    'message' => "Categories not found.",
                    'data'    => [
                        'items'        => [],
                        'total'        => 0,
                        'per_page'     => $per_page,
                        'current_page' => $page_number,
                        'last_page'    => 0,
                    ],
                    'status'  => 200,
                ];
            } else {
                $output = [
                    'success' => true,
                    'message' => "Categories found successfully.",
                    'data'    => [
                        'items'        => $categories->items(),
                        'total'        => $categories->total(),
                        'per_page'     => $categories->perPage(),
                        'current_page' => $categories->currentPage(),
                        'last_page'    => $categories->lastPage(),
                    ],
                    'status'  => 200,
                ];
            }

        } catch (\Exception $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());

            $output = [
                'success' => false,
                'message' => "Something went wrong, please try again: " . $e->getMessage(),
                'data'    => [
                    'items'        => [],
                    'total'        => 0,
                    'per_page'     => $per_page ?? 10,
                    'current_page' => $page_number ?? 1,
                    'last_page'    => 0,
                ],
                'status'  => 500,
            ];
        }

        return $output;
    }

    public function foodItemAdd(array $data) {
        try {
            $food_category_id = isset($data['food_category_id']) ? intval($data['food_category_id']) : 0;
            $food_item = isset($data['food_item']) ? trim($data['food_item']) : null;
            $title_description = isset($data['title_description']) ? trim($data['title_description']) : null;
            $long_description = isset($data['long_description']) ? trim($data['long_description']) : null;
            $ready_time = isset($data['ready_time']) ? intval($data['ready_time']) : 6;
            $weight = isset($data['weight']) ? intval($data['weight']) : 0;
            $price = isset($data['price']) ? floatval($data['price']) : 0.00;
            $main_image = isset($data['main_image']) ? trim($data['main_image']) : null;

            $images = isset($data['images']) && is_array($data['images']) ? $data['images'] : [];

            $country_id = isset($data['country_id']) ? intval($data['country_id']) : 0;
            $cut_type_id = isset($data['cut_type_id']) ? intval($data['cut_type_id']) : 0;
            $breed_id = isset($data['breed_id']) ? intval($data['breed_id']) : 0;

            $item_exists = FoodItem::whereRaw('LOWER(food_item) = ?', [strtolower($food_item)])
                                ->whereIn('is_active', [0, 1])
                                ->first();

            if (isset($item_exists->id)) {
                if ($item_exists->is_active == 0) {
                    $output['success'] = false;
                    $output['message'] = "Item already exists but is being deleted. food item id : " . $item_exists->id;
                    $output['data'] = null;
                    $output['status'] = 409;
                } else {
                    $output['success'] = false;
                    $output['message'] = "Item already exists.";
                    $output['data'] = null;
                    $output['status'] = 409;
                }
            } else {
                $category_exists = FoodCategory::where('id', $food_category_id)->where('is_active', 1)->first();
                if (!isset($category_exists->id)) {
                    $output['success'] = false;
                    $output['message'] = "Category does not exist.";
                    $output['data'] = null;
                    $output['status'] = 404;
                } else {
                    if ($food_category_id == 2) { // Meat Category
                        if ($country_id > 0 && $cut_type_id > 0) {
                            $country_exists = MeatCountry::where('id', $country_id)->where('is_active', 1)->first();
                            $cut_type_exists = MeatCutType::where('id', $cut_type_id)->where('is_active', 1)->first();
                            if ($breed_id > 0) {
                                $breed_exists = MeatBreed::where('id', $breed_id)->where('is_active', 1)->first();
                            }
                            if (!isset($country_exists->id) || !isset($cut_type_exists->id) || (!isset($breed_exists->id) && $breed_id > 0)) {
                                $output['success'] = false;
                                $output['message'] = "Country, Cut Type or Breed does not exist.";
                                $output['data'] = null;
                                $output['status'] = 404;
                                return $output;
                            }
                        } else {
                            $output['success'] = false;
                            $output['message'] = "Country, Cut Types are required for Meat Category.";
                            $output['data'] = null;
                            $output['status'] = 400;
                            return $output;
                        }
                    }

                    $date_time = Carbon::now();

                    $new_item = FoodItem::create([
                        'food_category_id' => $food_category_id,
                        'food_item' => ucwords(strtolower($food_item)),
                        'title_description' => $title_description,
                        'long_description' => $long_description,
                        'ready_time' => $ready_time,
                        'weight' => $weight,
                        'price' => $price,
                        'main_image' => $main_image,
                        'country_id' => $country_id,
                        'cut_type_id' => $cut_type_id,
                        'breed_id' => $breed_id,
                        'status' => 1,
                        'is_active' => 1,
                        'created_at' => $date_time,
                        'updated_at' => $date_time,
                    ]);
                    if(isset($new_item->id)) {
                        $category_exists->item_count += 1;
                        $category_exists->updated_at = $date_time;
                        $category_exists->save();

                        if (!empty($images)) {
                            foreach ($images as $image) {
                                DB::table('food_item_images')->insert([
                                    'food_item_id' => $new_item->id,
                                    'image' => $image,
                                    'is_active' => 1,
                                    'created_at' => $date_time,
                                    'updated_at' => $date_time
                                ]);
                            }
                        }

                        $output['success'] = true;
                        $output['message'] = "Item added successfully.";
                        $output['data']['item_id'] = $new_item->id;
                        $output['status'] = 201;
                    } else {
                        $output['success'] = false;
                        $output['message'] = "Item add failed!.";
                        $output['data'] = null;
                        $output['status'] = 400;
                        return $output;
                    }


                }
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodItemValidate(array $data) {
        try {
            $food_item_id = isset($data['food_item_id']) ? intval($data['food_item_id']) : 0;

            $item_exists = FoodItem::where('id', $food_item_id)->where('is_active', 1)->first();

            if (!isset($item_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Item does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $category = FoodCategory::where('id', $item_exists->food_category_id)->where('is_active', 1)->first();

                $item_images = DB::table('food_item_images')
                                ->where('food_item_id', $item_exists->id)
                                ->where('is_active', 1)
                                ->pluck('image')
                                ->toArray();

                $country = MeatCountry::where('id', $item_exists->country_id)->where('is_active', 1)->first();
                $cut_type = MeatCutType::where('id', $item_exists->cut_type_id)->where('is_active', 1)->first();
                $breed = MeatBreed::where('id', $item_exists->breed_id)->where('is_active', 1)->first();

                $output['success'] = true;
                $output['message'] = "Item exists.";
                $output['data']['id'] = isset($item_exists->id) ? intval($item_exists->id) : 0;
                $output['data']['food_category_id'] = isset($category->id) ? intval($category->id) : 0;
                $output['data']['food_category_name'] = isset($category->food_category) ? $category->food_category : null;
                $output['data']['food_item_name'] = isset($item_exists->food_item) ? $item_exists->food_item : null;
                $output['data']['title_description'] = isset($item_exists->title_description) ? $item_exists->title_description : null;
                $output['data']['long_description'] = isset($item_exists->long_description) ? $item_exists->long_description : null;
                $output['data']['ready_time'] = isset($item_exists->ready_time) ? $item_exists->ready_time : null;
                $output['data']['weight'] = isset($item_exists->weight) ? intval($item_exists->weight) : 0;
                $output['data']['price'] = isset($item_exists->price) ? floatval($item_exists->price) : 0.00;
                $output['data']['main_image'] = isset($item_exists->main_image) ? $item_exists->main_image : null;
                $output['data']['country_id'] = isset($item_exists->country_id) ? intval($item_exists->country_id) : 0;
                $output['data']['country_name'] = isset($country->country) ? $country->country : null;
                $output['data']['cut_type_id'] = isset($item_exists->cut_type_id) ? intval($item_exists->cut_type_id) : 0;
                $output['data']['cut_type_name'] = isset($cut_type->cut_type) ? $cut_type->cut_type : null;
                $output['data']['breed_id'] = isset($item_exists->breed_id) ? intval($item_exists->breed_id) : 0;
                $output['data']['breed_name'] = isset($breed->breed) ? $breed->breed : null;
                $output['data']['status'] = isset($item_exists->status) ? $item_exists->status : null;
                $output['data']['images'] = isset($item_images) ? $item_images : [];
                $output['status'] = 200;
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodItemEdit(array $data) {
        try {
            $food_item_id = isset($data['food_item_id']) ? intval($data['food_item_id']) : 0;
            $food_category_id = isset($data['food_category_id']) ? intval($data['food_category_id']) : 0;
            $food_item = isset($data['food_item']) ? trim($data['food_item']) : null;
            $title_description = isset($data['title_description']) ? trim($data['title_description']) : null;
            $long_description = isset($data['long_description']) ? trim($data['long_description']) : null;
            $ready_time = isset($data['ready_time']) ? intval($data['ready_time']) : 0;
            $weight = isset($data['weight']) ? intval($data['weight']) : 0;
            $price = isset($data['price']) ? floatval($data['price']) : 0.00;
            $main_image = isset($data['main_image']) ? trim($data['main_image']) : null;
            $status = isset($data['status']) ? intval($data['status']) : 0;

            $images = isset($data['images']) && is_array($data['images']) ? $data['images'] : [];

            $country_id = isset($data['country_id']) ? intval($data['country_id']) : 0;
            $cut_type_id = isset($data['cut_type_id']) ? intval($data['cut_type_id']) : 0;
            $breed_id = isset($data['breed_id']) ? intval($data['breed_id']) : 0;

            $item_exists = FoodItem::where('id', $food_item_id)/*->where('is_active', 1)*/->first();
            // Log::info($item_exists);
            if (!isset($item_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Item does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $date_time = Carbon::now();

                if ($food_category_id > 0 && $food_category_id != $item_exists->food_category_id) {
                    $new_category_exists = FoodCategory::where('id', $food_category_id)/*->where('is_active', 1)*/->first();
                    if (!isset($new_category_exists->id)) {
                        $output['success'] = false;
                        $output['message'] = "Category does not exist.";
                        $output['data'] = null;
                        $output['status'] = 404;
                        return $output;
                    }
                    $old_category = FoodCategory::where('id', $item_exists->food_category_id)->first();
                    $old_category->item_count -= 1;
                    $old_category->updated_at = $date_time;
                    //$old_category->save();

                    if ($country_id > 0 || $cut_type_id > 0) {
                        $category_exists = FoodCategory::where('id', $food_category_id)->first();
                        if ($food_category_id == 2) { // Meat Category
                            if ($country_id > 0) {
                                $country_exists = MeatCountry::where('id', $country_id)/*->where('is_active', 1)*/->first();
                            }
                            if ($cut_type_id > 0) {
                                $cut_type_exists = MeatCutType::where('id', $cut_type_id)/*->where('is_active', 1)*/->first();
                            }
                            if ($breed_id > 0) {
                                $breed_exists = MeatBreed::where('id', $breed_id)/*->where('is_active', 1)*/->first();
                            }
                            if ((!isset($country_exists->id) && $country_id > 0) || (!isset($cut_type_exists->id) && $cut_type_id > 0) || (!isset($breed_exists->id) && $breed_id > 0)) {
                                $output['success'] = false;
                                $output['message'] = "Country, Cut Type or Breed does not exist.";
                                $output['data'] = null;
                                $output['status'] = 404;
                                return $output;
                            } else {
                                $item_exists->country_id = $country_id;
                                $item_exists->cut_type_id = $cut_type_id;
                                $item_exists->breed_id = $breed_id;
                            }
                        } else {
                            $item_exists->country_id = 0;
                            $item_exists->cut_type_id = 0;
                            $item_exists->breed_id = 0;
                        }
                    }
                    $item_exists->food_category_id = $food_category_id;
                    $item_exists->updated_at = $date_time;
                    //$item_exists->save();
                    $new_category_exists->item_count += 1;
                    $new_category_exists->updated_at = $date_time;
                    //$new_category_exists->save();
                }
                // Log::info($item_exists);
                if ($food_item != null || $food_item != "") {
                    $item_name_exists = FoodItem::whereRaw('LOWER(food_item) = ?', [strtolower($food_item)])
                                                ->where('id', '!=', $food_item_id)
                                                /*->where('is_active', 1)*/
                                                ->exists();
                    if ($item_name_exists) {
                        $output['success'] = false;
                        $output['message'] = "Item name already exists.";
                        $output['data'] = null;
                        $output['status'] = 409;
                        return $output;
                    }
                    $item_exists->food_item = ucwords(strtolower($food_item));
                }
                if ($title_description != null || $title_description != "") {
                    $item_exists->title_description = $title_description;
                }
                if ($long_description != null || $long_description != "") {
                    $item_exists->long_description = $long_description;
                }
                if ($ready_time > 0) {
                    $item_exists->ready_time = $ready_time;
                }
                if ($weight > 0) {
                    $item_exists->weight = $weight;
                }
                if ($price > 0) {
                    $item_exists->price = $price;
                }
                if ($main_image != null || $main_image != "") {
                    $item_exists->main_image = $main_image;
                }

                if ($status == 0 || $status == 1) {
                    $item_exists->status = $status;
                }

                $item_exists->updated_at = $date_time;
                if(isset($old_category->id)) {
                    $old_category->save();
                }
                if(isset($new_category_exists->id)) {
                    $new_category_exists->save();
                }
                $item_exists->save();

                if (!empty($images)) {
                    DB::table('food_item_images')
                        ->where('food_item_id', $food_item_id)
                        ->where('is_active', 1)
                        ->update(['is_active' => 0]);

                    foreach ($images as $image) {
                        DB::table('food_item_images')->insert([
                            'food_item_id' => $food_item_id,
                            'image' => $image,
                            'is_active' => 1,
                            'created_at' => $date_time,
                            'updated_at' => $date_time
                        ]);
                    }

                }

                $output['success'] = true;
                $output['message'] = "Item updated successfully.";
                $output['data'] = null;
                $output['status'] = 200;
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodItemDelete(array $data) {
        try {
            $food_item_id = isset($data['food_item_id']) ? intval($data['food_item_id']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : 0;

            $item_exists = FoodItem::where('id', $food_item_id)->whereIn('is_active', [0, 1])->first();

            if (!isset($item_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Item does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $output['success'] = true;

                if ($item_exists->is_active != $status) {
                    if ($status != 1) {
                        $status = 0;
                        $output['message'] = "Item deleted successfully.";

                        FoodCategory::where('id', $item_exists->food_category_id)->decrement('item_count');
                    } else {
                        $output['message'] = "Item activated successfully.";

                        FoodCategory::where('id', $item_exists->food_category_id)->increment('item_count');
                    }
                    $date_time = Carbon::now();
                    $item_exists->is_active = $status;
                    $item_exists->updated_at = $date_time;
                    $item_exists->save();

                    DB::table('food_item_images')->where('food_item_id', $item_exists->id)->update(['is_active' => 0]);
                } else {
                    $output['message'] = "Item is already " . ($status == 1 ? "active." : "inactive.");
                }
                $output['data'] = null;
                $output['status'] = 200;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodItemAll(array $data)
    {
        try {
            $search           = $data['search']        ?? null;
            $status           = isset($data['status']) ? (int)$data['status'] : -1;      // accepts 0/1
            $is_active        = isset($data['is_active']) ? (int)$data['is_active'] : -1; // accepts 0/1
            $food_category_id = isset($data['food_category_id']) ? (int)$data['food_category_id'] : 0;
            $food_item_id = isset($data['food_item_id']) ? intval($data['food_item_id']) : 0;
            $page_number      = isset($data['page_number']) ? max(1, (int)$data['page_number']) : 1;
            $per_page         = isset($data['per_page']) ? max(1, (int)$data['per_page']) : 10;

            $query = DB::table('food_items as fi')
                ->leftJoin('food_categories as fc', 'fc.id', '=', 'fi.food_category_id')
                ->leftJoin('meat_countries as c', 'c.id', '=', 'fi.country_id')
                ->leftJoin('meat_cut_types as ct', 'ct.id', '=', 'fi.cut_type_id')
                ->leftJoin('meat_breeds as b', 'b.id', '=', 'fi.breed_id')
                ->select([
                    'fi.id',
                    'fi.food_item',
                    'fi.food_category_id',
                    'fc.food_category',
                    'fi.title_description',
                    'fi.long_description',
                    'fi.ready_time',
                    'fi.weight',
                    'fi.price',
                    'fi.country_id',
                    DB::raw('c.country as country_name'),
                    'fi.cut_type_id',
                    DB::raw('ct.cut_type as cut_type_name'),
                    'fi.breed_id',
                    DB::raw('b.breed as breed_name'),
                    'fi.status',
                    'fi.main_image',
                    'fi.is_active',
                    'fi.created_at',
                ])
                ->where('fi.id', '>', 0);

            // Filters
            if ($is_active === 0 || $is_active === 1) {
                $query->where('fi.is_active', $is_active);
            }
            if ($status === 0 || $status === 1) {
                $query->where('fi.status', $status);
            }
            if ($food_category_id > 0) {
                $query->where('fi.food_category_id', $food_category_id);
            }
            if ($food_item_id > 0) {
                $query->where('fi.id', $food_item_id);
            }
            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('fi.food_item', 'LIKE', "%{$search}%")
                    ->orWhere('fi.title_description', 'LIKE', "%{$search}%");
                });
            }

            $items = $query->orderBy('fi.created_at', 'DESC')
                ->paginate($per_page, ['*'], 'page', $page_number);

            // If no data, still return pagination meta
            if ($items->total() === 0) {
                return [
                    'success' => true,
                    'message' => 'Items not found.',
                    'data'    => [
                        'items'        => [],
                        'total'        => 0,
                        'per_page'     => $items->perPage(),
                        'current_page' => $items->currentPage(),
                        'last_page'    => $items->lastPage(),
                    ],
                    'status'  => 200,
                ];
            }

            // Fetch images for all items on this page
            $idsOnPage = collect($items->items())->pluck('id')->all();
            $imagesByItem = empty($idsOnPage) ? collect() :
                DB::table('food_item_images')
                    ->whereIn('food_item_id', $idsOnPage)
                    ->orderBy('id', 'ASC')
                    ->get(['food_item_id', 'image'])
                    ->groupBy('food_item_id')
                    ->map(fn($rows) => $rows->pluck('image')->values()->all());

            // Attach images (only here)
            $rows = [];
            foreach ($items as $row) {
                $rows[] = [
                    'id'                 => (int) ($row->id ?? 0),
                    'food_item_name'     => $row->food_item ?? null,
                    'food_category_id'   => (int) ($row->food_category_id ?? 0),
                    'food_category_name' => $row->food_category ?? null,
                    'title_description'  => $row->title_description ?? null,
                    'long_description'   => $row->long_description ?? null,
                    'ready_time'         => $row->ready_time ?? null,
                    'weight'             => isset($row->weight) ? (int)$row->weight : 0,
                    'price'              => isset($row->price) ? (float)$row->price : 0.0,
                    'country_id'         => (int) ($row->country_id ?? 0),
                    'country_name'       => $row->country_name ?? null,
                    'cut_type_id'        => (int) ($row->cut_type_id ?? 0),
                    'cut_type_name'      => $row->cut_type_name ?? null,
                    'breed_id'           => (int) ($row->breed_id ?? 0),
                    'breed_name'         => $row->breed_name ?? null,
                    'status'             => $row->status ?? null,
                    'main_image'         => $row->main_image ?? null,
                    'images'             => $imagesByItem->get($row->id, []),
                    'is_active'          => isset($row->is_active) ? (int)$row->is_active : 0,
                ];
            }

            return [
                'success' => true,
                'message' => 'Items found.',
                'data'    => [
                    'items'        => $rows,
                    'total'        => $items->total(),
                    'per_page'     => $items->perPage(),
                    'current_page' => $items->currentPage(),
                    'last_page'    => $items->lastPage(),
                ],
                'status'  => 200,
            ];
        } catch (\Throwable $e) {
            $url = $data['url'] ?? null;
            $this->logError($url, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Something went wrong, please try again: ' . $e->getMessage(),
                'data'    => [
                    'items'        => [],
                    'total'        => 0,
                    'per_page'     => $per_page ?? 10,
                    'current_page' => $page_number ?? 1,
                    'last_page'    => 0,
                ],
                'status'  => 500,
            ];
        }
    }

    public function foodItemStatusChange(array $data) {
        try {
            $food_item_id = isset($data['food_item_id']) ? intval($data['food_item_id']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : 0;

            $item_exists = FoodItem::where('id', $food_item_id)->where('is_active', 1)->first();

            if (!isset($item_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Item does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                if ($status == 0) {
                    $status_type = "Unavailable";
                } else {
                    $status_type = "Available";
                }
                if ($item_exists->status == $status) {
                    $output['success'] = false;
                    $output['message'] = "Item is already " . $status_type . ".";
                    $output['data'] = null;
                    $output['status'] = 409;
                } else {
                    $item_exists->status = $status;
                    $item_exists->updated_at = Carbon::now();
                    $item_exists->save();

                    $output['success'] = true;
                    $output['message'] = "Item status changed successfully.";
                    $output['data'] = null;
                    $output['status'] = 200;
                }
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCountryAdd(array $data) {
        try {
            $country = isset($data['country']) ? trim($data['country']) : null;

            $country_exists = MeatCountry::whereRaw('LOWER(country) = ?', [strtolower($country)])->where('is_active', 1)->first();

            if (isset($country_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Country already exists.";
                $output['data'] = null;
                $output['status'] = 409;
            } else {
                MeatCountry::create([
                    'country' => ucwords(strtolower($country)),
                    'is_active' => 1,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now()
                ]);

                $output['success'] = true;
                $output['message'] = "Country added successfully.";
                $output['data'] = null;
                $output['status'] = 201;
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCountryValidate(array $data) {
        try {
            $country_id = isset($data['country_id']) ? intval($data['country_id']) : 0;

            $country_exists = MeatCountry::where('id', $country_id)->where('is_active', 1)->first();

            $output['success'] = true;
            $output['status'] = 200;
            if (!isset($country_exists->id)) {
                $output['message'] = "Country does not exist.";
                $output['data'] = null;
            } else {
                $output['message'] = "Country exists.";
                $output['data']['id'] = isset($country_exists->id) ? intval($country_exists->id) : 0;
                $output['data']['country'] = isset($country_exists->country) ? $country_exists->country : null;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCountryAll(array $data) {
        try {
            $status = isset($data['status']) ? intval($data['status']) : -1;

            $query = MeatCountry::select('id', 'country', 'is_active AS status');

            if ($status == 1 ||  $status == 0) {
                if ($status == 1) {
                    $query->where('is_active', 1);
                } else {
                    $query->where('is_active', 0);
                }
            }

            $countries = $query->orderBy('id', 'asc')->get();

            if ($countries->isEmpty()) {
                $output['success'] = false;
                $output['message'] = "No countries found.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $output['success'] = true;
                $output['message'] = "Countries found.";
                $output['data'] = $countries;
                $output['status'] = 200;
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCountryDelete(array $data) {
        try {
            $country_id = isset($data['country_id']) ? intval($data['country_id']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : 0;

            $country_exists = MeatCountry::where('id', $country_id)->whereIn('is_active', [0, 1])->first();

            if (!isset($country_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Country does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $output['success'] = true;

                if ($country_exists->is_active != $status) {
                    if ($status != 1) {
                        $status = 0;
                        $output['message'] = "Country deleted successfully.";
                    } else {
                        $output['message'] = "Country activated successfully.";
                    }

                    $country_exists->is_active = $status;
                    $country_exists->updated_at = Carbon::now();
                    $country_exists->save();
                } else {
                    $output['message'] = "Country is already " . ($status == 1 ? "active." : "inactive.");
                }

                $output['data'] = null;
                $output['status'] = 200;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCutTypeAdd(array $data) {
        try {
            $cut_type = isset($data['cut_type']) ? trim($data['cut_type']) : null;

            $cut_type_exists = MeatCutType::whereRaw('LOWER(cut_type) = ?', [strtolower($cut_type)])->where('is_active', 1)->first();

            if (isset($cut_type_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Cut Type already exists.";
                $output['data'] = null;
                $output['status'] = 409;
            } else {
                MeatCutType::create([
                    'cut_type' => ucwords(strtolower($cut_type)),
                    'is_active' => 1,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now()
                ]);

                $output['success'] = true;
                $output['message'] = "Cut Type added successfully.";
                $output['data'] = null;
                $output['status'] = 201;
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCutTypeValidate(array $data) {
        try {
            $cut_type_id = isset($data['cut_type_id']) ? intval($data['cut_type_id']) : 0;

            $cut_type_exists = MeatCutType::where('id', $cut_type_id)->where('is_active', 1)->first();
            $output['status'] = 200;
            $output['success'] = true;
            if (!isset($cut_type_exists->id)) {
                $output['message'] = "Cut Type does not exist.";
                $output['data'] = null;
            } else {
                $output['message'] = "Cut Type exists.";
                $output['data']['id'] = isset($cut_type_exists->id) ? intval($cut_type_exists->id) : 0;
                $output['data']['cut_type'] = isset($cut_type_exists->cut_type) ? $cut_type_exists->cut_type : null;

            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCutTypeAll(array $data) {
        try {
            $status = isset($data['status']) ? intval($data['status']) : -1;

            $query = MeatCutType::select('id', 'cut_type', 'is_active AS status');

            if ($status == 1 ||  $status == 0) {
                if ($status == 1) {
                    $query->where('is_active', 1);
                } else {
                    $query->where('is_active', 0);
                }
            }

            $cut_types = $query->orderBy('id', 'ASC')->get();

            if ($cut_types->isEmpty()) {
                $output['success'] = false;
                $output['message'] = "No cut types found.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $output['success'] = true;
                $output['message'] = "Cut Types found.";
                $output['data'] = $cut_types;
                $output['status'] = 200;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatCutTypeDelete(array $data) {
        try {
            $cut_type_id = isset($data['cut_type_id']) ? intval($data['cut_type_id']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : 0;

            $cut_type_exists = MeatCutType::where('id', $cut_type_id)->whereIn('is_active', [0, 1])->first();

            if (!isset($cut_type_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Cut Type does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $output['success'] = true;

                if ($cut_type_exists->is_active != $status) {
                    if ($status != 1) {
                        $status = 0;
                        $output['message'] = "Cut Type deleted successfully.";
                    } else {
                        $output['message'] = "Cut Type activated successfully.";
                    }

                    $cut_type_exists->is_active = $status;
                    $cut_type_exists->updated_at = Carbon::now();
                    $cut_type_exists->save();
                } else {
                    $output['message'] = "Cut Type is already " . ($status == 1 ? "active." : "inactive.");
                }

                $output['data'] = null;
                $output['status'] = 200;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatBreedAdd(array $data) {
        try {
            $breed = isset($data['breed']) ? trim($data['breed']) : null;

            $breed_exists = MeatBreed::whereRaw('LOWER(breed) = ?', [strtolower($breed)])->where('is_active', 1)->first();

            if (isset($breed_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Breed already exists.";
                $output['data'] = null;
                $output['status'] = 409;
            } else {
                MeatBreed::create([
                    'breed' => ucwords(strtolower($breed)),
                    'is_active' => 1,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now()
                ]);

                $output['success'] = true;
                $output['message'] = "Breed added successfully.";
                $output['data'] = null;
                $output['status'] = 201;
            }

        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatBreedValidate(array $data) {
        try {
            $breed_id = isset($data['breed_id']) ? intval($data['breed_id']) : 0;

            $breed_exists = MeatBreed::where('id', $breed_id)->where('is_active', 1)->first();

            $output['status'] = 200;
            $output['success'] = true;

            if (!isset($breed_exists->id)) {
                $output['message'] = "Breed does not exist.";
                $output['data'] = null;
            } else {
                $output['message'] = "Breed exists.";
                $output['data']['id'] = isset($breed_exists->id) ? intval($breed_exists->id) : 0;
                $output['data']['breed'] = isset($breed_exists->breed) ? $breed_exists->breed : null;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatBreedAll(array $data) {
        try {
            $status = isset($data['status']) ? intval($data['status']) : -1;

            $query = MeatBreed::select('id', 'breed', 'is_active AS status');

            if ($status == 1 ||  $status == 0) {
                if ($status == 1) {
                    $query->where('is_active', 1);
                } else {
                    $query->where('is_active', 0);
                }
            }

            $breeds = $query->orderBy('id', 'asc')->get();

            if ($breeds->isEmpty()) {
                $output['success'] = true;
                $output['message'] = "Breeds not found.";
                $output['data'] = null;
                $output['status'] = 200;
            } else {
                $output['success'] = true;
                $output['message'] = "Breeds found.";
                $output['data'] = $breeds;
                $output['status'] = 200;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function foodMeatBreedDelete(array $data) {
        try {
            $breed_id = isset($data['breed_id']) ? intval($data['breed_id']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : 0;

            $breed_exists = MeatBreed::where('id', $breed_id)->whereIn('is_active', [0, 1])->first();

            if (!isset($breed_exists->id)) {
                $output['success'] = false;
                $output['message'] = "Breed does not exist.";
                $output['data'] = null;
                $output['status'] = 404;
            } else {
                $output['success'] = true;

                if ($breed_exists->is_active != $status) {
                    if ($status != 1) {
                        $status = 0;
                        $output['message'] = "Breed deleted successfully.";
                    } else {
                        $output['message'] = "Breed activated successfully.";
                    }

                    $breed_exists->is_active = $status;
                    $breed_exists->updated_at = Carbon::now();
                    $breed_exists->save();
                } else {
                    $output['message'] = "Breed is already " . ($status == 1 ? "active." : "inactive.");
                }

                $output['data'] = null;
                $output['status'] = 200;
            }
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }

    public function getBasicData(array $data) {
        try {
            $output['success'] = true;
            $output['message'] = "Basic data passed successfully.";
            $output['data']['food_category'] = FoodCategory::select('id', 'food_category', 'item_count',
                                                    'order_by', 'is_active', 'status')
                                                ->where('is_active', 1)->where('status', 1)
                                                ->orderBy('id', 'asc')->get();
            $output['data']['country'] = MeatCountry::select('id', 'country', 'is_active AS status')
                                            ->where('is_active', 1)
                                            ->orderBy('id', 'asc')->get();
            $output['data']['cut_type'] = MeatCutType::select('id', 'cut_type', 'is_active AS status')
                                            ->where('is_active', 1)
                                            ->orderBy('id', 'asc')->get();
            $output['data']['breed'] = MeatBreed::select('id', 'breed', 'is_active AS status')
                                        ->where('is_active', 1)
                                        ->orderBy('id', 'asc')->get();
            $output['status'] = 200;
        } catch (\Exception $e) {
            $url = isset($data['url']) ? $data['url'] : null;
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $output['status'] = 500;
        }
        return $output;
    }
}
