<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Models\Version;
use App\Models\Setting;
use Illuminate\Support\Facades\Log;
use Throwable;

class CheckAppVersion
{
    private function apiFail(string $message, array $data = [], int $status = 400)
    {
        return response()->json([
            'success' => false,
            'message' => $message,
            'data' => $data,
        ], $status);
    }

    public function handle(Request $request, Closure $next)
    {
        try {
            $buildNumber = (int) $request->header('Build-Number', 0);
            $appType     = (int) $request->header('App-Type', 0);
            $osType      = (int) $request->header('OS-Type', 0); // 0 = any, 1 = Android, 2 = iOS (example)

            // Maintenance Mode
            $maintainData = Setting::select('value')
                ->where('id', 1)
                ->where('is_active', 1)
                ->first();

            $maintainMode = isset($maintainData->value) ? (int) $maintainData->value : 0;

            if ($maintainMode === 1) {
                return $this->apiFail(
                    'Server is under maintenance.',
                    ['maintain_mode' => 1],
                    503
                );
            }

            // If no build number sent, allow request (or block if you want strict)
            if ($buildNumber <= 0) {
                return $next($request);
            }

            // If you want strict appType required, uncomment this
            // if ($appType <= 0) return $this->apiFail('Missing App-Type header.', [], 400);

            $query = Version::query()
                ->where('is_active', 1)
                ->where('app_type', $appType)
                ->where('build_number', '>', $buildNumber)
                ->where(function ($q) use ($osType) {
                    // os_type = 0 means "all OS"
                    if ($osType > 0) {
                        $q->where('os_type', 0)->orWhere('os_type', $osType);
                    } else {
                        $q->where('os_type', 0);
                    }
                })
                ->orderByDesc('id');

            $versionData = $query->first();

            if ($versionData) {
                // Better status for forcing update
                return $this->apiFail(
                    'A critical update is required to continue.',
                    ['has_update' => 1],
                    426 // Upgrade Required
                );
            }

            return $next($request);

        } catch (Throwable $e) {
            Log::error('Error in CheckAppVersion middleware', [
                'url' => $request->fullUrl(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            // fail safe: allow request OR block request (your choice)
            // I recommend BLOCK with 500, so you know something broke.
            return $this->apiFail('Internal server error.', [], 500);
        }
    }
}
