<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use App\Repositories\Interfaces\UserRepositoryInterface;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Exceptions\TokenExpiredException;
use Tymon\JWTAuth\Exceptions\TokenInvalidException;
use Illuminate\Support\Facades\Log; // Import Log facade at the top
use Illuminate\Validation\Rule;

use Tymon\JWTAuth\Facades\JWTAuth;

class UserController extends Controller
{
    private $userRepository;

    public function __construct(UserRepositoryInterface $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    protected function logError($url, $error_message)
    {
        Log::error('Error in user controller function', [
            'url' => $url,
            'error' => $error_message
        ]);
    }

    public function userRegister(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'role_id'       => 'required|integer',
                'full_name'     => 'required|string|min:1|max:255',
                'user_name'     => 'required|string|min:1|max:255|unique:users,user_name',
                'nic_number'    => 'required|string|min:1|max:255|unique:users,nic_number',
                'contact_number' => 'required|regex:/^\+?\d{10,25}$/|unique:users,contact_number',
                'password'      => 'required|string|min:6|max:256|confirmed',
                // optional inputs
                'push_id'       => 'nullable|string|max:255',
                'os_type'       => 'nullable|integer|in:0,1,2',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data'    => null,
                ], 422);
            }

            $data = $validator->validated();
            $data['url'] = $request->url();

            $out = $this->userRepository->userRegister($data);

            return response()->json([
                'success' => $out['success'],
                'message' => $out['message'],
                'data'    => $out['data'],
            ], $out['status']);

        } catch (\Throwable $e) {
            // If you have logError() you can call it here too
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

    public function userLogin(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_name' => 'required|string|min:1|max:255',
                'password'  => 'required|string|min:1|max:256',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data'    => null,
                ], 422);
            }

            $data = $validator->validated();
            $data['url'] = $request->url();

            $out = $this->userRepository->userLogin($data);

            return response()->json([
                'success' => $out['success'] ?? false,
                'message' => $out['message'] ?? 'Failed.',
                'data'    => $out['data'] ?? null,
            ], $out['status'] ?? 500);

        } catch (\Throwable $e) {
            $this->logError($request->url(), $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

    public function userValidate(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_name' => 'required|string|min:1|max:255',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data'    => null,
                ], 422);
            }

            $data = $validator->validated();
            $data['url'] = $request->url();

            $out = $this->userRepository->userValidate($data);

            return response()->json([
                'success' => $out['success'] ?? false,
                'message' => $out['message'] ?? 'Failed.',
                'data'    => $out['data'] ?? null,
            ], $out['status'] ?? 500);

        } catch (\Throwable $e) {
            $this->logError($request->url(), $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

    public function userData(Request $request)
    {
        try {
            $out = $this->userRepository->userData($request);

            return response()->json([
                'success' => $out['success'] ?? false,
                'message' => $out['message'] ?? 'Failed.',
                'data'    => $out['data'] ?? null,
            ], $out['status'] ?? 500);

        } catch (\Throwable $e) {
            $this->logError($request->url(), $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

    public function userAll(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_type_id' => 'nullable|integer',
                'search'       => 'nullable|string|min:1|max:255',
                'status'       => 'nullable|integer|in:0,1,-1',
                'page'         => 'nullable|integer|min:1',
                'per_page'     => 'nullable|integer|min:1|max:100',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data'    => null,
                ], 422);
            }

            $data = $validator->validated();
            $data['url'] = $request->url();

            $out = $this->userRepository->userAll($data);

            return response()->json([
                'success' => $out['success'] ?? false,
                'message' => $out['message'] ?? 'Failed.',
                'data'    => $out['data'] ?? null,
            ], $out['status'] ?? 500);

        } catch (\Throwable $e) {
            $this->logError($request->url(), $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

    public function userManage(Request $request)
    {
        try {
            $id = $request->input('id');

            $validator = Validator::make($request->all(), [
                'user_id'            => 'required|integer|exists:users,id',

                'full_name'     => 'required|string|min:1|max:255',

                'nic_number'    => [
                    'required','string','min:1','max:255',
                    Rule::unique('users', 'nic_number')->ignore($user_id),
                ],

                'mobile_number' => [
                    'required',
                    'regex:/^\+?\d{10,25}$/',
                    Rule::unique('users', 'mobile_number')->ignore($user_id),
                ],

                'email_address' => [
                    'required','string','email:rfc,dns','max:255',
                    Rule::unique('users', 'email_address')->ignore($user_id),
                ],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data'    => null,
                ], 422);
            }

            $data = $validator->validated();
            $data['url'] = $request->url();

            $out = $this->userRepository->userManage($data);

            return response()->json([
                'success' => $out['success'],
                'message' => $out['message'],
                'data'    => $out['data'],
            ], $out['status']);

        } catch (\Throwable $e) {
            $this->logError($request->url(), $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

    public function userPasswordChange(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'current_password' => 'required|string|min:1|max:256',
                'new_password'     => 'required|string|min:6|max:256|confirmed',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data'    => null,
                ], 422);
            }

            $data = $validator->validated();
            $data['url'] = $request->url();

            $out = $this->userRepository->adminPasswordChange($data);

            return response()->json([
                'success' => $out['success'],
                'message' => $out['message'],
                'data'    => $out['data'],
            ], $out['status']);

        } catch (\Throwable $e) {
            $this->logError($request->url(), $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

    public function userLogout(Request $request)
    {
        try {
            $data['url'] = $request->url();

            $out = $this->userRepository->userLogout($data);

            return response()->json([
                'success' => $out['success'],
                'message' => $out['message'],
                'data'    => $out['data'],
            ], $out['status']);

        } catch (\Throwable $e) {
            $this->logError($request->url(), $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong, please try again.',
                'data'    => null,
            ], 500);
        }
    }

}
