<?php

namespace App\Http\Controllers;

use App\Models\FoodCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use App\Repositories\Interfaces\SettingRepositoryInterface;
use Illuminate\Support\Facades\Log; // Import Log facade at the top


class SettingController extends Controller
{
    private $settingRepository;

    public function __construct(SettingRepositoryInterface $settingRepository)
    {
        $this->settingRepository = $settingRepository;
    }
    protected function logError($url, $error_message)
    {
        Log::error('Error in user controller function', [
            'url' => $url,
            'error' => $error_message
        ]);
    }

    public function foodCategoryAdd(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_category' => 'required|string|min:1|max:255|unique:food_categories,food_category',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodCategoryAdd($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodCategoryValidate(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_category_id' => 'required|integer|exists:food_categories,id',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodCategoryValidate($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodCategoryEdit(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_category_id' => 'required|integer|exists:food_categories,id',
                'food_category' => 'required|string|min:1|max:255'
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodCategoryEdit($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodCategoryDelete(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_category_id' => 'required|integer|exists:food_categories,id',
                'status' => 'nullable|integer|in:0,1'
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodCategoryDelete($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodCategoryAll(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'is_active' => 'nullable|in:0,1,-1',
                'search' => 'nullable|string|min:1|max:255',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodCategoryAll($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodItemAdd(Request $request) {
        try {
            $baseRules = [
                'food_category_id' => 'required|integer|exists:food_categories,id',
                'food_item' => 'nullable|string|min:1|max:255|unique:food_items,food_item',
                'title_description' => 'nullable|string|min:1|max:255',
                'long_description' => 'nullable|string|min:1|max:2048',
                'ready_time' => 'nullable|integer|in:1,2,3,4,5,6,7,8,9,10,11',
                'weight' => 'required|integer|min:1',
                'price' => 'required|numeric|min:0',
                'main_image' => [
                    'nullable', 'string',
                    'regex:/^(https?:\/\/)?[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}(\/[\w\-\.~:\/?#[\]@!$&\'()*+,;=%]*)?\.(jpg|jpeg|png|gif)(\?.*)?$/i'
                ],
                'images' => 'nullable|array|min:2|max:5',
                'image.*' => [
                    'nullable', 'string',
                    'regex:/^(https?:\/\/)?[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}(\/[\w\-\.~:\/?#[\]@!$&\'()*+,;=%]*)?\.(jpg|jpeg|png|gif)(\?.*)?$/i'
                ],
            ];

            $category = FoodCategory::find($request->food_category_id);

            if ($request->food_category_id == 2) {
                $meatRules = [
                    'country_id' => 'required|integer|exists:meat_countries,id',
                    'cut_type_id' => 'required|integer|exists:meat_cut_types,id',
                    'breed_id' => 'nullable|integer|exists:meat_breeds,id',
                ];

                $baseRules = array_merge($baseRules, $meatRules);
            } /*else {
                if ($request->has('country_id') || $request->has('cut_type_id') || $request->has('breed_id')) {
                    $output['success'] = false;
                    $output['message'] = "Country, Cut Type and Breed are not valid for this category.";
                    $output['data'] = null;
                    $status = 422;
                    return response()->json([
                        'success' => $output['success'],
                        'message' => $output['message'],
                        'output' => $output['data']
                    ], $status);
                }
            }*/

            $validator = Validator::make($request->all(), $baseRules);

            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodItemAdd($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodItemValidate(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_item_id' => 'required|integer|exists:food_items,id',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodItemValidate($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodItemEdit(Request $request) {
        try {
            $baseRules = [
                'food_item_id' => 'required|integer|exists:food_items,id',
                'food_category_id' => 'nullable|integer|exists:food_categories,id',
                'food_item' => 'nullable|string|min:1|max:255',
                'title_description' => 'nullable|string|min:1|max:255',
                'long_description' => 'nullable|string|min:1|max:2048',
                'ready_time' => 'nullable|integer|in:1,2,3,4,5,6,7,8,9,10,11',
                'weight' => 'nullable|integer|min:1',
                'price' => 'nullable|numeric|min:0',
                // 'main_image' => 'nullable|file|mimes:jpg,jpeg,png|max:10240',
                'main_image' => [
                    'nullable', 'string',
                    'regex:/^(https?:\/\/)?[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}(\/[\w\-\.~:\/?#[\]@!$&\'()*+,;=%]*)?\.(jpg|jpeg|png|gif)(\?.*)?$/i'
                ],
                'images' => 'nullable|array|min:2|max:5',
                'image.*' => [
                    'nullable', 'string',
                    'regex:/^(https?:\/\/)?[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}(\/[\w\-\.~:\/?#[\]@!$&\'()*+,;=%]*)?\.(jpg|jpeg|png|gif)(\?.*)?$/i'
                ],
                // 'main_image' => 'nullable|file|mimes:jpg,jpeg,png|max:10240',
                // 'image_gallery' => 'nullable|array|min:2|max:5',
                // 'image_gallery.*' => 'file|mimes:jpg,jpeg,png|max:10240',
            ];
            $category = FoodCategory::find($request->food_category_id);
            // dd($category);

            if ($request->food_category_id == 2) {
                $meatRules = [
                    'country_id' => 'required|integer|exists:meat_countries,id',
                    'cut_type_id' => 'required|integer|exists:meat_cut_types,id',
                    'breed_id' => 'nullable|integer|exists:meat_breeds,id',
                ];

                $baseRules = array_merge($baseRules, $meatRules);
            } /*else {
                if ($request->has('country_id') || $request->has('cut_type_id') || $request->has('breed_id')) {
                    $output['success'] = false;
                    $output['message'] = "Country, Cut Type and Breed are not valid for this category.";
                    $output['data'] = null;
                    $status = 422;
                    return response()->json([
                        'success' => $output['success'],
                        'message' => $output['message'],
                        'output' => $output['data']
                    ], $status);
                }
            }*/

            $validator = Validator::make($request->all(), $baseRules);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodItemEdit($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodItemDelete(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_item_id' => 'required|integer|exists:food_items,id',
                'is_active' => 'nullable|integer|in:0,1',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodItemDelete($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodItemAll(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_category_id' => 'nullable|integer',
                'is_active' => 'nullable|integer|in:0,1,-1',
                'status' => 'nullable|integer|in:0,1,-1',
                'search' => 'nullable|string|min:1|max:255',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodItemAll($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodItemStatusChange(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'food_item_id' => 'required|integer|exists:food_items,id',
                'status' => 'required|integer|in:0,1'
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodItemStatusChange($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCountryAdd(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'country' => 'required|string|min:1|max:255',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCountryAdd($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCountryValidate(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'country_id' => 'required|integer|exists:meat_countries,id',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCountryValidate($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCountryAll(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'is_active' => 'nullable|integer|in:0,1,-1',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCountryAll($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCountryDelete(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'country_id' => 'required|integer|exists:meat_countries,id',
                'status' => 'nullable|integer|in:0,1',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCountryDelete($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCutTypeAdd(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'cut_type' => 'required|string|min:1|max:255',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCutTypeAdd($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCutTypeValidate(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'cut_type_id' => 'required|integer|exists:meat_cut_types,id',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCutTypeValidate($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCutTypeAll(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'is_active' => 'nullable|integer|in:0,1,-1',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCutTypeAll($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatCutTypeDelete(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'cut_type_id' => 'required|integer|exists:meat_cut_types,id',
                'status' => 'nullable|integer|in:0,1',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatCutTypeDelete($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatBreedAdd(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'breed' => 'required|string|min:1|max:255',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatBreedAdd($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatBreedValidate(Request $request ) {
        try {
            $validator = Validator::make($request->all(), [
                'breed_id' => 'required|integer|exists:meat_breeds,id',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatBreedValidate($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatBreedAll(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'is_active' => 'nullable|integer|in:0,1,-1',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatBreedAll($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function foodMeatBreedDelete(Request $request) {
        try {
            $validator = Validator::make($request->all(), [
                'breed_id' => 'required|integer|exists:meat_breeds,id',
                'status' => 'nullable|integer|in:0,1',
            ]);
            if ($validator->fails()) {
                $output['success'] = false;
                $output['message'] = $validator->errors()->first();
                $output['data'] = null;
                $status = 422;
            } else {
                $data = json_decode($request->getContent(), true);
                $data['url'] = $request->url();
                $out_data = $this->settingRepository->foodMeatBreedDelete($data);

                $output['success'] = $out_data['success'];
                $output['message'] = $out_data['message'];
                $output['data'] = $out_data['data'];
                $status = $out_data['status'];
            }
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

    public function getBasicData(Request $request) {
        try {
            $data = json_decode($request->getContent(), true);
            $data['url'] = $request->url();
            $out_data = $this->settingRepository->getBasicData($data);

            $output['success'] = $out_data['success'];
            $output['message'] = $out_data['message'];
            $output['data'] = $out_data['data'];
            $status = $out_data['status'];
        } catch (\Exception $e) {
            $url = $request->url();
            $error_message = $e->getMessage();
            $this->logError($url, $error_message);
            $output['success'] = false;
            $output['message'] = "Something went wrong, please try again: " . $e->getMessage();
            $output['data'] = null;
            $status = 500;
        }
        return response()->json(['success' => $output['success'], 'message' => $output['message'], 'output' => $output['data']], $status);
    }

}
